<?php

declare (strict_types=1);
namespace WPForms\Vendor\Square\Models;

use stdClass;
/**
 * Square Checkout lets merchants accept online payments for supported
 * payment types using a checkout workflow hosted on squareup.com.
 */
class Checkout implements \JsonSerializable
{
    /**
     * @var string|null
     */
    private $id;
    /**
     * @var array
     */
    private $checkoutPageUrl = [];
    /**
     * @var array
     */
    private $askForShippingAddress = [];
    /**
     * @var array
     */
    private $merchantSupportEmail = [];
    /**
     * @var array
     */
    private $prePopulateBuyerEmail = [];
    /**
     * @var Address|null
     */
    private $prePopulateShippingAddress;
    /**
     * @var array
     */
    private $redirectUrl = [];
    /**
     * @var Order|null
     */
    private $order;
    /**
     * @var string|null
     */
    private $createdAt;
    /**
     * @var array
     */
    private $additionalRecipients = [];
    /**
     * Returns Id.
     * ID generated by Square Checkout when a new checkout is requested.
     */
    public function getId() : ?string
    {
        return $this->id;
    }
    /**
     * Sets Id.
     * ID generated by Square Checkout when a new checkout is requested.
     *
     * @maps id
     */
    public function setId(?string $id) : void
    {
        $this->id = $id;
    }
    /**
     * Returns Checkout Page Url.
     * The URL that the buyer's browser should be redirected to after the
     * checkout is completed.
     */
    public function getCheckoutPageUrl() : ?string
    {
        if (\count($this->checkoutPageUrl) == 0) {
            return null;
        }
        return $this->checkoutPageUrl['value'];
    }
    /**
     * Sets Checkout Page Url.
     * The URL that the buyer's browser should be redirected to after the
     * checkout is completed.
     *
     * @maps checkout_page_url
     */
    public function setCheckoutPageUrl(?string $checkoutPageUrl) : void
    {
        $this->checkoutPageUrl['value'] = $checkoutPageUrl;
    }
    /**
     * Unsets Checkout Page Url.
     * The URL that the buyer's browser should be redirected to after the
     * checkout is completed.
     */
    public function unsetCheckoutPageUrl() : void
    {
        $this->checkoutPageUrl = [];
    }
    /**
     * Returns Ask for Shipping Address.
     * If `true`, Square Checkout will collect shipping information on your
     * behalf and store that information with the transaction information in your
     * Square Dashboard.
     *
     * Default: `false`.
     */
    public function getAskForShippingAddress() : ?bool
    {
        if (\count($this->askForShippingAddress) == 0) {
            return null;
        }
        return $this->askForShippingAddress['value'];
    }
    /**
     * Sets Ask for Shipping Address.
     * If `true`, Square Checkout will collect shipping information on your
     * behalf and store that information with the transaction information in your
     * Square Dashboard.
     *
     * Default: `false`.
     *
     * @maps ask_for_shipping_address
     */
    public function setAskForShippingAddress(?bool $askForShippingAddress) : void
    {
        $this->askForShippingAddress['value'] = $askForShippingAddress;
    }
    /**
     * Unsets Ask for Shipping Address.
     * If `true`, Square Checkout will collect shipping information on your
     * behalf and store that information with the transaction information in your
     * Square Dashboard.
     *
     * Default: `false`.
     */
    public function unsetAskForShippingAddress() : void
    {
        $this->askForShippingAddress = [];
    }
    /**
     * Returns Merchant Support Email.
     * The email address to display on the Square Checkout confirmation page
     * and confirmation email that the buyer can use to contact the merchant.
     *
     * If this value is not set, the confirmation page and email will display the
     * primary email address associated with the merchant's Square account.
     *
     * Default: none; only exists if explicitly set.
     */
    public function getMerchantSupportEmail() : ?string
    {
        if (\count($this->merchantSupportEmail) == 0) {
            return null;
        }
        return $this->merchantSupportEmail['value'];
    }
    /**
     * Sets Merchant Support Email.
     * The email address to display on the Square Checkout confirmation page
     * and confirmation email that the buyer can use to contact the merchant.
     *
     * If this value is not set, the confirmation page and email will display the
     * primary email address associated with the merchant's Square account.
     *
     * Default: none; only exists if explicitly set.
     *
     * @maps merchant_support_email
     */
    public function setMerchantSupportEmail(?string $merchantSupportEmail) : void
    {
        $this->merchantSupportEmail['value'] = $merchantSupportEmail;
    }
    /**
     * Unsets Merchant Support Email.
     * The email address to display on the Square Checkout confirmation page
     * and confirmation email that the buyer can use to contact the merchant.
     *
     * If this value is not set, the confirmation page and email will display the
     * primary email address associated with the merchant's Square account.
     *
     * Default: none; only exists if explicitly set.
     */
    public function unsetMerchantSupportEmail() : void
    {
        $this->merchantSupportEmail = [];
    }
    /**
     * Returns Pre Populate Buyer Email.
     * If provided, the buyer's email is pre-populated on the checkout page
     * as an editable text field.
     *
     * Default: none; only exists if explicitly set.
     */
    public function getPrePopulateBuyerEmail() : ?string
    {
        if (\count($this->prePopulateBuyerEmail) == 0) {
            return null;
        }
        return $this->prePopulateBuyerEmail['value'];
    }
    /**
     * Sets Pre Populate Buyer Email.
     * If provided, the buyer's email is pre-populated on the checkout page
     * as an editable text field.
     *
     * Default: none; only exists if explicitly set.
     *
     * @maps pre_populate_buyer_email
     */
    public function setPrePopulateBuyerEmail(?string $prePopulateBuyerEmail) : void
    {
        $this->prePopulateBuyerEmail['value'] = $prePopulateBuyerEmail;
    }
    /**
     * Unsets Pre Populate Buyer Email.
     * If provided, the buyer's email is pre-populated on the checkout page
     * as an editable text field.
     *
     * Default: none; only exists if explicitly set.
     */
    public function unsetPrePopulateBuyerEmail() : void
    {
        $this->prePopulateBuyerEmail = [];
    }
    /**
     * Returns Pre Populate Shipping Address.
     * Represents a postal address in a country.
     * For more information, see [Working with Addresses](https://developer.squareup.com/docs/build-
     * basics/working-with-addresses).
     */
    public function getPrePopulateShippingAddress() : ?Address
    {
        return $this->prePopulateShippingAddress;
    }
    /**
     * Sets Pre Populate Shipping Address.
     * Represents a postal address in a country.
     * For more information, see [Working with Addresses](https://developer.squareup.com/docs/build-
     * basics/working-with-addresses).
     *
     * @maps pre_populate_shipping_address
     */
    public function setPrePopulateShippingAddress(?Address $prePopulateShippingAddress) : void
    {
        $this->prePopulateShippingAddress = $prePopulateShippingAddress;
    }
    /**
     * Returns Redirect Url.
     * The URL to redirect to after checkout is completed with `checkoutId`,
     * Square's `orderId`, `transactionId`, and `referenceId` appended as URL
     * parameters. For example, if the provided redirect_url is
     * `http://www.example.com/order-complete`, a successful transaction redirects
     * the customer to:
     *
     * <pre><code>http://www.example.com/order-complete?checkoutId=xxxxxx&amp;orderId=xxxxxx&amp;
     * referenceId=xxxxxx&amp;transactionId=xxxxxx</code></pre>
     *
     * If you do not provide a redirect URL, Square Checkout will display an order
     * confirmation page on your behalf; however Square strongly recommends that
     * you provide a redirect URL so you can verify the transaction results and
     * finalize the order through your existing/normal confirmation workflow.
     */
    public function getRedirectUrl() : ?string
    {
        if (\count($this->redirectUrl) == 0) {
            return null;
        }
        return $this->redirectUrl['value'];
    }
    /**
     * Sets Redirect Url.
     * The URL to redirect to after checkout is completed with `checkoutId`,
     * Square's `orderId`, `transactionId`, and `referenceId` appended as URL
     * parameters. For example, if the provided redirect_url is
     * `http://www.example.com/order-complete`, a successful transaction redirects
     * the customer to:
     *
     * <pre><code>http://www.example.com/order-complete?checkoutId=xxxxxx&amp;orderId=xxxxxx&amp;
     * referenceId=xxxxxx&amp;transactionId=xxxxxx</code></pre>
     *
     * If you do not provide a redirect URL, Square Checkout will display an order
     * confirmation page on your behalf; however Square strongly recommends that
     * you provide a redirect URL so you can verify the transaction results and
     * finalize the order through your existing/normal confirmation workflow.
     *
     * @maps redirect_url
     */
    public function setRedirectUrl(?string $redirectUrl) : void
    {
        $this->redirectUrl['value'] = $redirectUrl;
    }
    /**
     * Unsets Redirect Url.
     * The URL to redirect to after checkout is completed with `checkoutId`,
     * Square's `orderId`, `transactionId`, and `referenceId` appended as URL
     * parameters. For example, if the provided redirect_url is
     * `http://www.example.com/order-complete`, a successful transaction redirects
     * the customer to:
     *
     * <pre><code>http://www.example.com/order-complete?checkoutId=xxxxxx&amp;orderId=xxxxxx&amp;
     * referenceId=xxxxxx&amp;transactionId=xxxxxx</code></pre>
     *
     * If you do not provide a redirect URL, Square Checkout will display an order
     * confirmation page on your behalf; however Square strongly recommends that
     * you provide a redirect URL so you can verify the transaction results and
     * finalize the order through your existing/normal confirmation workflow.
     */
    public function unsetRedirectUrl() : void
    {
        $this->redirectUrl = [];
    }
    /**
     * Returns Order.
     * Contains all information related to a single order to process with Square,
     * including line items that specify the products to purchase. `Order` objects also
     * include information about any associated tenders, refunds, and returns.
     *
     * All Connect V2 Transactions have all been converted to Orders including all associated
     * itemization data.
     */
    public function getOrder() : ?Order
    {
        return $this->order;
    }
    /**
     * Sets Order.
     * Contains all information related to a single order to process with Square,
     * including line items that specify the products to purchase. `Order` objects also
     * include information about any associated tenders, refunds, and returns.
     *
     * All Connect V2 Transactions have all been converted to Orders including all associated
     * itemization data.
     *
     * @maps order
     */
    public function setOrder(?Order $order) : void
    {
        $this->order = $order;
    }
    /**
     * Returns Created At.
     * The time when the checkout was created, in RFC 3339 format.
     */
    public function getCreatedAt() : ?string
    {
        return $this->createdAt;
    }
    /**
     * Sets Created At.
     * The time when the checkout was created, in RFC 3339 format.
     *
     * @maps created_at
     */
    public function setCreatedAt(?string $createdAt) : void
    {
        $this->createdAt = $createdAt;
    }
    /**
     * Returns Additional Recipients.
     * Additional recipients (other than the merchant) receiving a portion of this checkout.
     * For example, fees assessed on the purchase by a third party integration.
     *
     * @return AdditionalRecipient[]|null
     */
    public function getAdditionalRecipients() : ?array
    {
        if (\count($this->additionalRecipients) == 0) {
            return null;
        }
        return $this->additionalRecipients['value'];
    }
    /**
     * Sets Additional Recipients.
     * Additional recipients (other than the merchant) receiving a portion of this checkout.
     * For example, fees assessed on the purchase by a third party integration.
     *
     * @maps additional_recipients
     *
     * @param AdditionalRecipient[]|null $additionalRecipients
     */
    public function setAdditionalRecipients(?array $additionalRecipients) : void
    {
        $this->additionalRecipients['value'] = $additionalRecipients;
    }
    /**
     * Unsets Additional Recipients.
     * Additional recipients (other than the merchant) receiving a portion of this checkout.
     * For example, fees assessed on the purchase by a third party integration.
     */
    public function unsetAdditionalRecipients() : void
    {
        $this->additionalRecipients = [];
    }
    /**
     * Encode this object to JSON
     *
     * @param bool $asArrayWhenEmpty Whether to serialize this model as an array whenever no fields
     *        are set. (default: false)
     *
     * @return array|stdClass
     */
    #[\ReturnTypeWillChange]
    public function jsonSerialize(bool $asArrayWhenEmpty = \false)
    {
        $json = [];
        if (isset($this->id)) {
            $json['id'] = $this->id;
        }
        if (!empty($this->checkoutPageUrl)) {
            $json['checkout_page_url'] = $this->checkoutPageUrl['value'];
        }
        if (!empty($this->askForShippingAddress)) {
            $json['ask_for_shipping_address'] = $this->askForShippingAddress['value'];
        }
        if (!empty($this->merchantSupportEmail)) {
            $json['merchant_support_email'] = $this->merchantSupportEmail['value'];
        }
        if (!empty($this->prePopulateBuyerEmail)) {
            $json['pre_populate_buyer_email'] = $this->prePopulateBuyerEmail['value'];
        }
        if (isset($this->prePopulateShippingAddress)) {
            $json['pre_populate_shipping_address'] = $this->prePopulateShippingAddress;
        }
        if (!empty($this->redirectUrl)) {
            $json['redirect_url'] = $this->redirectUrl['value'];
        }
        if (isset($this->order)) {
            $json['order'] = $this->order;
        }
        if (isset($this->createdAt)) {
            $json['created_at'] = $this->createdAt;
        }
        if (!empty($this->additionalRecipients)) {
            $json['additional_recipients'] = $this->additionalRecipients['value'];
        }
        $json = \array_filter($json, function ($val) {
            return $val !== null;
        });
        return !$asArrayWhenEmpty && empty($json) ? new stdClass() : $json;
    }
}
